//////////////////////////////////////////////////////////////////////////////
//  Copyright 2014 by Autodesk, Inc.  All rights reserved.
//
// This computer source code and related instructions and comments 
// are the unpublished confidential and proprietary information of 
// Autodesk, Inc. and are protected under applicable copyright and 
// trade secret law.  They may not be disclosed to, copied or used 
// by any third party without the prior written consent of Autodesk, Inc.
//////////////////////////////////////////////////////////////////////////////

/*globals $,jQuery,document, window, navigator*/

/*!
 * ntp.page - Implements animations and functionality of new tab page elements
 */

 // Namespace
var ntp = ntp || {};
ntp.page = ntp.page || {};

ntp.page.animationSettings = {
    TAB_SWITCHING_ANIMATION_DURATION: 600,
    ELEMENT_TOGGLING_ANIMATION_DURATION: 200
}

/**
 * Handles page layout that cannot be done via CSS
 */
ntp.page.layout = (function () {
    var $tabCreateLearn = null;
    var $columns = null;
    var $getStartedVideo = null;
    var $featuredVideo = null;
    var $onlineResource = null;
    var $continueWorking = null;
    var $connectPanel = null;
    var columnMinHeight = 504;
    //This is video item height. thumb height 90px, thumb border is 2px and thumb bottom-margin is 8px.
    var videoItemHeight = 100;
    //This is video item margin. video item margin-bottom is 16px.
    var videoItemGap = 16;
    
   
    var getColumnHeight = function() {
        var newHeight = $tabCreateLearn.height() * 0.7;
        if (newHeight < columnMinHeight){
            newHeight = columnMinHeight;
        }
        return newHeight;
    };
    
    var adjustTopSpace = function() {
        var newColumnHeight = $tabCreateLearn.height() * 0.7;
        var space = $tabCreateLearn.height() - newColumnHeight;
        if (space < 0)
            space = 0;
        $tabCreateLearn.children('.content').css('margin-top', 0.375 * space);
    };
    
    return {
        init: function () {
            $canvas = $('#canvas');
            $tabs = $('#canvas_slider > .tab');
            $tabCreateLearn = $('#tab_create, #tab_learn');
            $columns = $('section.column');
            $getStartedVideo = $('#getting_started ol');
            $featuredVideo = $('#feature_videos ol');
            $continueWorking = $('#continue_working');
            $onlineResource = $('#online_resources_panel');
            $connectPanel = $('#connect_scroll_panel');
            this.adjustColumnHeight();
        },
        
        adjustColumnHeight: function () {
            var newHeight = getColumnHeight();
            $columns.height(newHeight);
            $continueWorking.height(newHeight);
            this.updateVideoSectionHeight();
            this.updateOnlineResourceHeight();
            this.updateConnectHeight();
        },
        
        update: function () {
            // Update tab width
            var tabWidth = $canvas.width();
            $tabs.each(function () {
                $(this).width(tabWidth);
            });
            
            this.adjustColumnHeight();
            adjustTopSpace();
        },
        
        updateVideoSectionHeight: function() {
            var height = getColumnHeight();
            if(ntp.content.learnTab.hasGetStartedVideos() && ntp.content.learnTab.hasFeaturedVideos()) {
                var getStartedItemsHeight = $getStartedVideo.children().length * videoItemHeight
                                                + ($getStartedVideo.children().length - 1) * videoItemGap;
                var featuredItemsHeight = $featuredVideo.children().length * videoItemHeight
                                                + ($featuredVideo.children().length - 1) * videoItemGap;
                var getStartedHeight = height * 0.5;
                if (getStartedHeight > getStartedItemsHeight)
                    getStartedHeight = getStartedItemsHeight;
                var featuredHeight = height - getStartedHeight
                                        - $('#getting_started_title').outerHeight(true)
                                        - $('#video_section_spacing').outerHeight(true)
                                        - $('#featured_title').outerHeight(true);
                if (featuredHeight > featuredItemsHeight){
                    featuredHeight = featuredItemsHeight;
                    getStartedHeight = height - featuredHeight
                                        - $('#getting_started_title').outerHeight(true)
                                        - $('#video_section_spacing').outerHeight(true)
                                        - $('#featured_title').outerHeight(true);
                }
                $getStartedVideo.css({'height': 'auto', 'max-height': getStartedHeight});
                $featuredVideo.css({'height': 'auto','max-height': featuredHeight});
            }
            else if(ntp.content.learnTab.hasGetStartedVideos()) {
                var getStartedHeight = height - $('#getting_started_title').outerHeight(true);
                $getStartedVideo.css({'height': 'auto','max-height': getStartedHeight});
            }
            else if(ntp.content.learnTab.hasFeaturedVideos()) {
                var featuredHeight = height - $('#featured_title').outerHeight(true);
                $featuredVideo.css({'height': 'auto', 'max-height': featuredHeight});
            }
        },
        
        updateOnlineResourceHeight: function() {
            if(ntp.content.learnTab.hasOnlineResource()) {
                var height = getColumnHeight();
                var tipHeight = 0;
                if(ntp.content.learnTab.hasTips()) {
                    tipHeight = $('#Section_tip').outerHeight(true);
                }
                var onlineHeight = height - $('#online_resource_title').outerHeight(true) - tipHeight;
                $onlineResource.css('max-height', onlineHeight);
            }
        },
        
        updateConnectHeight: function() {
            if(ntp.widget.connect.hasConnect()){
                var height = getColumnHeight();
                var panelHeight;
                if(ntp.widget.notifications.hasNotifications()){
                    panelHeight = height - $('#connect_title').outerHeight(true) - $('#section_notifications').outerHeight(true);
                }
                else {
                    panelHeight = height - $('#connect_title').outerHeight(true);
                }
                $connectPanel.css('max-height', panelHeight);
            }
        }
    }
})();

/**
 * Handles tabs and tab switching
 */
ntp.page.tabs = (function () {
    
    var $canvas = null;
    var $tabs = null;
    var tipContentsVisited = false;
    var selectedIndex = 2;
    
    return {
        current: function () {
            return selectedIndex;
        },

        isTipContentsVisited: function () {
            return tipContentsVisited;
        },
        
        count: function () {
            if ($tabs) {
                return $tabs.length;
            } else {
                return 0;
            }
        },
        
        init: function () {
            $canvas = $('#canvas');
            $tabs = $('#canvas_slider > .tab');
        },
        
        // Selects a tab by its index
        selectByIndex: function(index, immediate, forced) {
            var indexChanged = true;
            if (selectedIndex === index) {
                if (!forced) { return; }
                indexChanged = false;
            }
            
            selectedIndex = index;

            // Updates footer navigator class
            $('#footer_nav li').each(function (index) {
                var tabIndex = index + 1;
                if (tabIndex === selectedIndex) {
                    $(this).addClass('highlight');
                } else {
                    $(this).removeClass('highlight');
                }
            });

            // If 'immediate' is auto, then an animation will be used only if canvas is already running an animation.
            if (immediate === 'auto') {
                immediate = !$canvas.is(':animated');
            }

            // Updates tab position
            var $tab = $('.tab:nth-child(' + index + ')');

            if (tipContentsVisited === true && ntp.content.learnTab.urlIsHilightedContent('tip')) {
                //Leaving learn tab, so unhilight tip
                ntp.content.learnTab.unhilightViewedLearnConents({ tip: $('div#tip_resource_box').data('upload_time') });
                tipContentsVisited = false;

            }else if ($tab.attr('id') === 'tab_learn' && ntp.content.learnTab.urlIsHilightedContent('tip')) {

                //First time Visiting, so notify all others except self.
                ntp.acad.updateLearnContentState('tip', $('div#tip_resource_box').data('upload_time'));
                tipContentsVisited = true;
            }

            var newScrollLeft = $tab.position().left;
            if ($canvas.scrollLeft() !== newScrollLeft) {
                if (immediate) {
                    $canvas.stop().scrollLeft(newScrollLeft);
                } else {
                    var durationMultiplier = Math.abs(newScrollLeft - $canvas.scrollLeft()) / $canvas.width();
                    if (durationMultiplier > 1) { durationMultiplier = 1; }
                    $canvas.stop().animate(
                        { scrollLeft: newScrollLeft },
                        durationMultiplier * ntp.page.animationSettings.TAB_SWITCHING_ANIMATION_DURATION,
                        'linear');
                }
            }
            
            ntp.page.sideNavs.update(immediate, true);
        },
        
        update: function () {
            this.selectByIndex(selectedIndex, 'auto', true);
        },

        // Selects a tab by its id
        selectById: function(id, immediate, forced) {
            var tabIndex = $('.tab').index($('#' + id));
            if (tabIndex >= 0) {
                this.selectByIndex(tabIndex + 1, immediate, forced);
            }
        },
        
        // Auto snaps to the closest tab after scrolling
        snap: function (immediate) {
            var tabWidth = $canvas.width();
            var currentScrollLeft = $canvas.scrollLeft();
            
            // Return if scrolling is caused by resizing
            if (currentScrollLeft === (tabWidth * (selectedIndex - 1))) {
                return;
            }
            
            var closestTabIndex = Math.floor((currentScrollLeft + tabWidth / 2) / tabWidth) + 1;
            this.selectByIndex(closestTabIndex, immediate, true);
        }
    }
})();

/**
 * Handles side navigation arrows
 */
ntp.page.sideNavs = (function () {
    
    var $canvas = null;
    var $tabContents = null;
    var $sideNavs = null;
    
    var tabContentWidth = null;
    var leftSideNavWidth = null;
    var rightSideNavWidth = null;
    
    var overlapped = null;
    
    var currentTab = null;
    
    // Updates the global variable storing left and right of the content of currently selected tab        
    var updateCurrentTabContentWidth = function () {
        if ($tabContents === null) { return null; }
        var selectedIndex = ntp.page.tabs.current();
        var $content = $tabContents.eq(selectedIndex - 1);
        var $columns = $content.children('.column');
        if ($columns.length === 0) {
            tabContentWidth = $content.outerWidth(true);
        } else {
            var $firstColumn = $columns.eq(0);
            var $lastColumn = $columns.eq($columns.length - 1);
            tabContentWidth = $lastColumn.offset().left + $lastColumn.outerWidth(true) - $firstColumn.offset().left;
        }
    }
    
    // Updates side navigator overlap status. If side navigators overlap with page content, they would be hidden.
    var updateOverlap = function (immediate) {
        // Update stored tab content width
        updateCurrentTabContentWidth();

        var tabWidth = $canvas.width();
        var sideNavSpace = (tabWidth - tabContentWidth) / 2;
    
        var newOverlapped = ((sideNavSpace - leftSideNavWidth) < 10 || 
                (sideNavSpace -rightSideNavWidth) < 10);

        if (newOverlapped !== overlapped) {
            if (overlapped === null || immediate) {
                if (newOverlapped) {
                    $sideNavs.addClass('overlapped');
                } else {
                    $sideNavs.removeClass('overlapped');
                }
            } else {
                // Use a fadein or fadeout animation when showing or hiding the side navigators
                $sideNavs.stop(true);
                if (newOverlapped) {
                    $sideNavs.css('opacity', 1);
                    $sideNavs.animate({ opacity: 0 }, 
                        ntp.page.animationSettings.TAB_SWITCHING_ANIMATION_DURATION / 2, 
                        'linear', function () {
                        $sideNavs.css('opacity', '');
                        $sideNavs.addClass('overlapped');
                    });
                } else {
                    $sideNavs.css('opacity', 0);
                    $sideNavs.removeClass('overlapped');
                    $sideNavs.animate({ opacity: 1 }, 
                        ntp.page.animationSettings.TAB_SWITCHING_ANIMATION_DURATION / 2, 
                        'linear', function () {
                        $sideNavs.css('opacity', '');
                    });
                }
            }

            overlapped = newOverlapped;
        }
    }
    
    // Updates total width of side navigation tools in global variable.
    var updateWidth = function () {
        leftSideNavWidth = 33;
        leftSideNavWidth += $sideNavs.eq(0).outerWidth(true);
        rightSideNavWidth = 33;
        rightSideNavWidth += $sideNavs.eq(1).outerWidth(true);    
    };
    
    // Updates individual side navigator according to currently selected tab.
    var updateNavigator = function (navigatorSelector, navigatorTabIndex, immediate, handleOverlap) {
        var $sideNav = $(navigatorSelector);

        $sideNav.off('click'); // Clear existing event handler

        if (immediate) {
            var $footerNav = $('#footer_nav li:nth-child(' + navigatorTabIndex + ') .label');
            var sideNavShouldShow = ($footerNav.length > 0);
            if (sideNavShouldShow) {
                // Target tab exists
                $sideNav.removeClass('disabled')
                        .children('.label')
                        .text($footerNav.text());

                $sideNav.on('click', function (event) {
                    ntp.page.tabs.selectByIndex(navigatorTabIndex);
                    event.preventDefault();
                });
            } else {
                // Target tab does not exist
                $sideNav.addClass('disabled');
            }

            // Update width of side navigators after their size has changed
            updateWidth();

            if (handleOverlap) {
                updateOverlap(true);
            }
        } else {
            // Change navigators with fade in/fade out effect. The implementation is a little bit complicated,
            // as the simple jQuery fadeIn() fadeOut() do not work so well here.
            var fadeIn = function () {
                $sideNav.stop().css('opacity', '');
                updateNavigator(navigatorSelector, navigatorTabIndex, true, handleOverlap);
                if ($sideNav.css('display') !== 'none') {
                    $sideNav.animate({ opacity: 1 }, 
                        ntp.page.animationSettings.TAB_SWITCHING_ANIMATION_DURATION / 2, 
                        'linear', function () {
                        $sideNav.css('opacity', '');
                    });
                }
            };

            $sideNav.stop();
            if ($sideNav.css('display') === 'none') {
                setTimeout(fadeIn, ntp.page.animationSettings.TAB_SWITCHING_ANIMATION_DURATION / 2);
            } else {
                $sideNav.animate({ opacity: 0 }, 
                    ntp.page.animationSettings.TAB_SWITCHING_ANIMATION_DURATION / 2, 
                    'linear', fadeIn);
            }
        }
    }
    
    return {
        init: function () {
            $canvas = $('#canvas');         
            $tabContents = $('#canvas_slider > .tab > .content');
            $sideNavs = $('#side_nav li');
        },
        
        // Updates both side navigators according to currently selected tab.
        update: function (immediate, handleOverlap, forced) {
            var newCurrentTab = ntp.page.tabs.current();
            
            var tabChanged = false;
            if (currentTab != newCurrentTab) {
                tabChanged = true;
                currentTab = newCurrentTab;
            }
            
            if (tabChanged || forced) {
                // Update side navigators
                updateNavigator('#left_side_nav', currentTab - 1, immediate, handleOverlap);
                updateNavigator('#right_side_nav', currentTab + 1, immediate, handleOverlap);
            } else {
                updateOverlap();
            }
        }
    }
})();

/**
 * Handles footer navigation buttons
 */
ntp.page.footerNavs = (function () {
    var $canvas = null;
    var $footerNavs = null;
    var $navArrow = null;
    
    var footerNavWidth = null;
    
    return {
        init: function () {
            $canvas = $('#canvas');
            $footerNavs = $('#footer_nav li');
            $navArrow = $('#nav_arrow');
            
            $('#footer_nav .button').each(function (index) {
                var tabIndex = index + 1;   // Index used in CSS selector starts from 1.
                $(this).click(function (event) {
                    ntp.page.tabs.selectByIndex(tabIndex);
                    event.preventDefault();
                });
            });
        
            // Makes the footer navs equal width, and store footerNavWidth
            footerNavWidth = 0;
            var innerWidth = 0;
            $footerNavs.each(function () {
                $(this).width('auto');
                var currentNavWidth = $(this).width();
                if (currentNavWidth > innerWidth) {
                    innerWidth = currentNavWidth;
                }
            });
            
            $footerNavs.each(function () {
                $(this).width(innerWidth);
            });

            if ($footerNavs.length > 1) {
                // Calculate width from offset of two elements. This is needed to count in a mysterious space between the elements in the width
                // (elements already have margin = 0 )
                var firstTabLeft = $footerNavs.eq(0).offset().left;
                var secondTabLeft = $footerNavs.eq(1).offset().left;
                if (firstTabLeft === secondTabLeft && firstTabLeft > 0)
                    footerNavWidth = $footerNavs.eq(0).innerWidth();
                else
                    footerNavWidth = $footerNavs.eq(1).offset().left - $footerNavs.eq(0).offset().left;
            } else {
                footerNavWidth = $(this).outerWidth(true);
            }
        },
        
        update: function () {
            if (!footerNavWidth) { // footerNavWidth might be 0 if the window size is too small
                this.init();
            }
            var tabWidth = $canvas.width();
            var currentScrollLeft = $canvas.scrollLeft();
            var percentageLocation = (currentScrollLeft + tabWidth / 2) / tabWidth;
            var percentageIndex = Math.floor(percentageLocation);
            var footerFirstTab = $footerNavs.eq(0);
            var footerLeft = footerFirstTab.offset().left + footerFirstTab.innerWidth() * percentageIndex;
            var newOffset = footerLeft + footerNavWidth * (percentageLocation - percentageIndex) - $navArrow.width() / 2;
            $navArrow.css('left', newOffset);
        }
    };
})();


/**
 * Page initialization on customization complete
 */
ntp.deferred.customization.done(function () {
    ntp.page.layout.init();
    ntp.page.tabs.init();
    ntp.widget.templatebar.init();
    
    if (ntp.page.tabs.count() > 1) {
        ntp.page.sideNavs.init();
        
        $('#canvas').scroll(function () {
            // Synchronize position of navigation arrow with scroll left.
            ntp.page.footerNavs.update();
        });
    }

    $(window).resize(function() {
        ntp.widget.templatebar.update();
        ntp.widget.projectbar.update();
        ntp.page.layout.update();

        if (ntp.page.tabs.count() > 1) {
            ntp.page.footerNavs.update();
            // Fix the problem of window resizing when switch tab transition is not finished.
            ntp.page.tabs.update();
        }
    });

    $(document).on('keydown', function(e) {
        if (e.ctrlKey || e.altKey) {
            ntp.acad.notifyShortcutKeyPressed(e);
        }
    });
    
    $(document).mousedown(function(e) {
        if(e.button == 1) {
            e.preventDefault();
        }
    });

    // Initialize layouts
    ntp.page.layout.update();
    
    // Show create tab on start
    if (ntp.page.tabs.count() > 1) {	
        ntp.page.tabs.selectById('tab_create', true, true);
    }
});

/**
 * Page initialization on customization and localization complete.
 */
$.when(ntp.deferred.localization, ntp.deferred.customization).done(function () {
    // Updates layout after text is loaded. Some elements are resized automatically.
    if (ntp.page.tabs.count() > 1) {
        ntp.page.footerNavs.init();
        ntp.page.sideNavs.update(true, true, true); // Update side navigator names
    }
});

/**
 * Allows only dropping dwg files
 */
(function () {
    function isDrawingFile(path) {
        path = path || "";
        path = path.substr(path.length - 4).toLowerCase();
        return (path == ".dxf") || (path == ".dwg") || (path == ".dws") || (path == ".dwt");
    }

    function queryDropHandler(e) {
        e.preventDefault();
        e.stopPropagation();

        var data = e.originalEvent.dataTransfer;
        data.dropEffect = "none";
        if (data && data.files) {
            for (var i = 0; i < data.files.length; ++i) {
                if (isDrawingFile(data.files[i].name)) {
                    data.dropEffect = "copy";
                    break;
                }
            }
        }
    }

    $(window).on('dragenter', queryDropHandler);
    $(window).on('dragover', queryDropHandler);
    $(window).on('drop', function(e) {
        e.preventDefault();
        e.stopPropagation();
        ntp.acad.openDroppedFiles();
    });
})();

// SIG // Begin signature block
// SIG // MIIZNgYJKoZIhvcNAQcCoIIZJzCCGSMCAQExCzAJBgUr
// SIG // DgMCGgUAMGcGCisGAQQBgjcCAQSgWTBXMDIGCisGAQQB
// SIG // gjcCAR4wJAIBAQQQEODJBs441BGiowAQS9NQkAIBAAIB
// SIG // AAIBAAIBAAIBADAhMAkGBSsOAwIaBQAEFI4/Emf+GznX
// SIG // ikQmgjBAuQQ0p5OMoIIUMDCCA+4wggNXoAMCAQICEH6T
// SIG // 6/t8xk5Z6kuad9QG/DswDQYJKoZIhvcNAQEFBQAwgYsx
// SIG // CzAJBgNVBAYTAlpBMRUwEwYDVQQIEwxXZXN0ZXJuIENh
// SIG // cGUxFDASBgNVBAcTC0R1cmJhbnZpbGxlMQ8wDQYDVQQK
// SIG // EwZUaGF3dGUxHTAbBgNVBAsTFFRoYXd0ZSBDZXJ0aWZp
// SIG // Y2F0aW9uMR8wHQYDVQQDExZUaGF3dGUgVGltZXN0YW1w
// SIG // aW5nIENBMB4XDTEyMTIyMTAwMDAwMFoXDTIwMTIzMDIz
// SIG // NTk1OVowXjELMAkGA1UEBhMCVVMxHTAbBgNVBAoTFFN5
// SIG // bWFudGVjIENvcnBvcmF0aW9uMTAwLgYDVQQDEydTeW1h
// SIG // bnRlYyBUaW1lIFN0YW1waW5nIFNlcnZpY2VzIENBIC0g
// SIG // RzIwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIB
// SIG // AQCxrLNJVEuXHBIK2CV5kSJXKm/cuCbEQ3Nrwr8uUFr7
// SIG // FMJ2jkMBJUO0oeJF9Oi3e8N0zCLXtJQAAvdN7b+0t0Qk
// SIG // a81fRTvRRM5DEnMXgotptCvLmR6schsmTXEfsTHd+1Fh
// SIG // AlOmqvVJLAV4RaUvic7nmef+jOJXPz3GktxK+Hsz5HkK
// SIG // +/B1iEGc/8UDUZmq12yfk2mHZSmDhcJgFMTIyTsU2sCB
// SIG // 8B8NdN6SIqvK9/t0fCfm90obf6fDni2uiuqm5qonFn1h
// SIG // 95hxEbziUKFL5V365Q6nLJ+qZSDT2JboyHylTkhE/xni
// SIG // RAeSC9dohIBdanhkRc1gRn5UwRN8xXnxycFxAgMBAAGj
// SIG // gfowgfcwHQYDVR0OBBYEFF+a9W5czMx0mtTdfe8/2+xM
// SIG // gC7dMDIGCCsGAQUFBwEBBCYwJDAiBggrBgEFBQcwAYYW
// SIG // aHR0cDovL29jc3AudGhhd3RlLmNvbTASBgNVHRMBAf8E
// SIG // CDAGAQH/AgEAMD8GA1UdHwQ4MDYwNKAyoDCGLmh0dHA6
// SIG // Ly9jcmwudGhhd3RlLmNvbS9UaGF3dGVUaW1lc3RhbXBp
// SIG // bmdDQS5jcmwwEwYDVR0lBAwwCgYIKwYBBQUHAwgwDgYD
// SIG // VR0PAQH/BAQDAgEGMCgGA1UdEQQhMB+kHTAbMRkwFwYD
// SIG // VQQDExBUaW1lU3RhbXAtMjA0OC0xMA0GCSqGSIb3DQEB
// SIG // BQUAA4GBAAMJm495739ZMKrvaLX64wkdu0+CBl03X6ZS
// SIG // nxaN6hySCURu9W3rWHww6PlpjSNzCxJvR6muORH4KrGb
// SIG // sBrDjutZlgCtzgxNstAxpghcKnr84nodV0yoZRjpeUBi
// SIG // JZZux8c3aoMhCI5B6t3ZVz8dd0mHKhYGXqY4aiISo1EZ
// SIG // g362MIIEozCCA4ugAwIBAgIQDs/0OMj+vzVuBNhqmBsa
// SIG // UDANBgkqhkiG9w0BAQUFADBeMQswCQYDVQQGEwJVUzEd
// SIG // MBsGA1UEChMUU3ltYW50ZWMgQ29ycG9yYXRpb24xMDAu
// SIG // BgNVBAMTJ1N5bWFudGVjIFRpbWUgU3RhbXBpbmcgU2Vy
// SIG // dmljZXMgQ0EgLSBHMjAeFw0xMjEwMTgwMDAwMDBaFw0y
// SIG // MDEyMjkyMzU5NTlaMGIxCzAJBgNVBAYTAlVTMR0wGwYD
// SIG // VQQKExRTeW1hbnRlYyBDb3Jwb3JhdGlvbjE0MDIGA1UE
// SIG // AxMrU3ltYW50ZWMgVGltZSBTdGFtcGluZyBTZXJ2aWNl
// SIG // cyBTaWduZXIgLSBHNDCCASIwDQYJKoZIhvcNAQEBBQAD
// SIG // ggEPADCCAQoCggEBAKJjCzlEuLsjp0RJuw7/ofBhClOT
// SIG // sJjbrSwPSsVu/4Y8U1UPFc4EPyv9qZaW2b5heQtbyUyG
// SIG // duXgQ0sile7CK0PBn9hotI5AT+6FOLkRxSPyZFjwFTJv
// SIG // TlehroikAtcqHs1L4d1j1ReJMluwXplaqJ0oUA4X7pbb
// SIG // YTtFUR3PElYLkkf8q672Zj1HrHBy55LnX80QucSDZJQZ
// SIG // vSWA4ejSIqXQugJ6oXeTW2XD7hd0vEGGKtwITIySjJEt
// SIG // nndEH2jWqHR32w5bMotWizO92WPISZ06xcXqMwvS8aMb
// SIG // 9Iu+2bNXizveBKd6IrIkri7HcMW+ToMmCPsLvalPmQjh
// SIG // EChyqs0CAwEAAaOCAVcwggFTMAwGA1UdEwEB/wQCMAAw
// SIG // FgYDVR0lAQH/BAwwCgYIKwYBBQUHAwgwDgYDVR0PAQH/
// SIG // BAQDAgeAMHMGCCsGAQUFBwEBBGcwZTAqBggrBgEFBQcw
// SIG // AYYeaHR0cDovL3RzLW9jc3Aud3Muc3ltYW50ZWMuY29t
// SIG // MDcGCCsGAQUFBzAChitodHRwOi8vdHMtYWlhLndzLnN5
// SIG // bWFudGVjLmNvbS90c3MtY2EtZzIuY2VyMDwGA1UdHwQ1
// SIG // MDMwMaAvoC2GK2h0dHA6Ly90cy1jcmwud3Muc3ltYW50
// SIG // ZWMuY29tL3Rzcy1jYS1nMi5jcmwwKAYDVR0RBCEwH6Qd
// SIG // MBsxGTAXBgNVBAMTEFRpbWVTdGFtcC0yMDQ4LTIwHQYD
// SIG // VR0OBBYEFEbGaaMOShQe1UzaUmMXP142vA3mMB8GA1Ud
// SIG // IwQYMBaAFF+a9W5czMx0mtTdfe8/2+xMgC7dMA0GCSqG
// SIG // SIb3DQEBBQUAA4IBAQB4O7SRKgBM8I9iMDd4o4QnB28Y
// SIG // st4l3KDUlAOqhk4ln5pAAxzdzuN5yyFoBtq2MrRtv/Qs
// SIG // JmMz5ElkbQ3mw2cO9wWkNWx8iRbG6bLfsundIMZxD82V
// SIG // dNy2XN69Nx9DeOZ4tc0oBCCjqvFLxIgpkQ6A0RH83Vx2
// SIG // bk9eDkVGQW4NsOo4mrE62glxEPwcebSAe6xp9P2ctgwW
// SIG // K/F/Wwk9m1viFsoTgW0ALjgNqCmPLOGy9FqpAa8VnCwv
// SIG // SRvbIrvD/niUUcOGsYKIXfA9tFGheTMrLnu53CAJE3Hr
// SIG // ahlbz+ilMFcsiUk/uc9/yb8+ImhjU5q9aXSsxR08f5Lg
// SIG // w7wc2AR1MIIFhTCCBG2gAwIBAgIQKcFbP6rNUmpOZ708
// SIG // Tn4/8jANBgkqhkiG9w0BAQUFADCBtDELMAkGA1UEBhMC
// SIG // VVMxFzAVBgNVBAoTDlZlcmlTaWduLCBJbmMuMR8wHQYD
// SIG // VQQLExZWZXJpU2lnbiBUcnVzdCBOZXR3b3JrMTswOQYD
// SIG // VQQLEzJUZXJtcyBvZiB1c2UgYXQgaHR0cHM6Ly93d3cu
// SIG // dmVyaXNpZ24uY29tL3JwYSAoYykxMDEuMCwGA1UEAxMl
// SIG // VmVyaVNpZ24gQ2xhc3MgMyBDb2RlIFNpZ25pbmcgMjAx
// SIG // MCBDQTAeFw0xMjA3MjUwMDAwMDBaFw0xNTA5MjAyMzU5
// SIG // NTlaMIHIMQswCQYDVQQGEwJVUzETMBEGA1UECBMKQ2Fs
// SIG // aWZvcm5pYTETMBEGA1UEBxMKU2FuIFJhZmFlbDEWMBQG
// SIG // A1UEChQNQXV0b2Rlc2ssIEluYzE+MDwGA1UECxM1RGln
// SIG // aXRhbCBJRCBDbGFzcyAzIC0gTWljcm9zb2Z0IFNvZnR3
// SIG // YXJlIFZhbGlkYXRpb24gdjIxHzAdBgNVBAsUFkRlc2ln
// SIG // biBTb2x1dGlvbnMgR3JvdXAxFjAUBgNVBAMUDUF1dG9k
// SIG // ZXNrLCBJbmMwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAw
// SIG // ggEKAoIBAQCoYmDrmd0Gq8ezSsDlfgaJFEFplNPNhWzM
// SIG // 2uFQaYAB/ggpQ11+N4B6ao+TqrNIWDIqt3JKhaU889nx
// SIG // l/7teWGwuOurstI2Z0bEDhXiXam/bicK2HVLyntliQ+6
// SIG // tT+nlgfN8tgB2NzM0BpE1YCnU2b6DwQw4V7BV+/F//83
// SIG // yGFOpePlumzXxNw9EKWkaq81slmmTxf7UxZgP9PGbLw8
// SIG // gLAPk4PTJI97+5BBqhkLb1YqSfWn3PNMfsNKhw/VwAN0
// SIG // dRKeM6H8SkOdz+osr+NyH86lsKQuics4fwK5uFSHQHsI
// SIG // t6Z0tqWvminRqceUi9ugRlGryh9X1ZqCqfL/ggdzYa3Z
// SIG // AgMBAAGjggF7MIIBdzAJBgNVHRMEAjAAMA4GA1UdDwEB
// SIG // /wQEAwIHgDBABgNVHR8EOTA3MDWgM6Axhi9odHRwOi8v
// SIG // Y3NjMy0yMDEwLWNybC52ZXJpc2lnbi5jb20vQ1NDMy0y
// SIG // MDEwLmNybDBEBgNVHSAEPTA7MDkGC2CGSAGG+EUBBxcD
// SIG // MCowKAYIKwYBBQUHAgEWHGh0dHBzOi8vd3d3LnZlcmlz
// SIG // aWduLmNvbS9ycGEwEwYDVR0lBAwwCgYIKwYBBQUHAwMw
// SIG // cQYIKwYBBQUHAQEEZTBjMCQGCCsGAQUFBzABhhhodHRw
// SIG // Oi8vb2NzcC52ZXJpc2lnbi5jb20wOwYIKwYBBQUHMAKG
// SIG // L2h0dHA6Ly9jc2MzLTIwMTAtYWlhLnZlcmlzaWduLmNv
// SIG // bS9DU0MzLTIwMTAuY2VyMB8GA1UdIwQYMBaAFM+Zqep7
// SIG // JvRLyY6P1/AFJu/j0qedMBEGCWCGSAGG+EIBAQQEAwIE
// SIG // EDAWBgorBgEEAYI3AgEbBAgwBgEBAAEB/zANBgkqhkiG
// SIG // 9w0BAQUFAAOCAQEA2OkGvuiY7TyI6yVTQAYmTO+MpOFG
// SIG // C8MflHSbofJiuLxrS1KXbkzsAPFPPsU1ouftFhsXFtDQ
// SIG // 8rMTq/jwugTpbJUREV0buEkLl8AKRhYQTKBKg1I/puBv
// SIG // bkJocDE0pRwtBz3xSlXXEwyYPcbCOnrM3OZ5bKx1Qiii
// SIG // vixlcGWhO3ws904ssutPFf4mV5PDi3U2Yp1HgbBK/Um/
// SIG // FLr6YAYeZaA8KY1CfQEisF3UKTwm72d7S+fJf++SOGea
// SIG // K0kumehVcbavQJTOVebuZ9V+qU0nk1lMrqve9BnQK69B
// SIG // QqNZu77vCO0wm81cfynAxkOYKZG3idY47qPJOgXKkwmI
// SIG // 2+92ozCCBgowggTyoAMCAQICEFIA5aolVvwahu2WydRL
// SIG // M8cwDQYJKoZIhvcNAQEFBQAwgcoxCzAJBgNVBAYTAlVT
// SIG // MRcwFQYDVQQKEw5WZXJpU2lnbiwgSW5jLjEfMB0GA1UE
// SIG // CxMWVmVyaVNpZ24gVHJ1c3QgTmV0d29yazE6MDgGA1UE
// SIG // CxMxKGMpIDIwMDYgVmVyaVNpZ24sIEluYy4gLSBGb3Ig
// SIG // YXV0aG9yaXplZCB1c2Ugb25seTFFMEMGA1UEAxM8VmVy
// SIG // aVNpZ24gQ2xhc3MgMyBQdWJsaWMgUHJpbWFyeSBDZXJ0
// SIG // aWZpY2F0aW9uIEF1dGhvcml0eSAtIEc1MB4XDTEwMDIw
// SIG // ODAwMDAwMFoXDTIwMDIwNzIzNTk1OVowgbQxCzAJBgNV
// SIG // BAYTAlVTMRcwFQYDVQQKEw5WZXJpU2lnbiwgSW5jLjEf
// SIG // MB0GA1UECxMWVmVyaVNpZ24gVHJ1c3QgTmV0d29yazE7
// SIG // MDkGA1UECxMyVGVybXMgb2YgdXNlIGF0IGh0dHBzOi8v
// SIG // d3d3LnZlcmlzaWduLmNvbS9ycGEgKGMpMTAxLjAsBgNV
// SIG // BAMTJVZlcmlTaWduIENsYXNzIDMgQ29kZSBTaWduaW5n
// SIG // IDIwMTAgQ0EwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAw
// SIG // ggEKAoIBAQD1I0tepdeKuzLp1Ff37+THJn6tGZj+qJ19
// SIG // lPY2axDXdYEwfwRof8srdR7NHQiM32mUpzejnHuA4Jnh
// SIG // 7jdNX847FO6G1ND1JzW8JQs4p4xjnRejCKWrsPvNamKC
// SIG // TNUh2hvZ8eOEO4oqT4VbkAFPyad2EH8nA3y+rn59wd35
// SIG // BbwbSJxp58CkPDxBAD7fluXF5JRx1lUBxwAmSkA8taEm
// SIG // qQynbYCOkCV7z78/HOsvlvrlh3fGtVayejtUMFMb32I0
// SIG // /x7R9FqTKIXlTBdOflv9pJOZf9/N76R17+8V9kfn+Bly
// SIG // 2C40Gqa0p0x+vbtPDD1X8TDWpjaO1oB21xkupc1+NC2J
// SIG // AgMBAAGjggH+MIIB+jASBgNVHRMBAf8ECDAGAQH/AgEA
// SIG // MHAGA1UdIARpMGcwZQYLYIZIAYb4RQEHFwMwVjAoBggr
// SIG // BgEFBQcCARYcaHR0cHM6Ly93d3cudmVyaXNpZ24uY29t
// SIG // L2NwczAqBggrBgEFBQcCAjAeGhxodHRwczovL3d3dy52
// SIG // ZXJpc2lnbi5jb20vcnBhMA4GA1UdDwEB/wQEAwIBBjBt
// SIG // BggrBgEFBQcBDARhMF+hXaBbMFkwVzBVFglpbWFnZS9n
// SIG // aWYwITAfMAcGBSsOAwIaBBSP5dMahqyNjmvDz4Bq1EgY
// SIG // LHsZLjAlFiNodHRwOi8vbG9nby52ZXJpc2lnbi5jb20v
// SIG // dnNsb2dvLmdpZjA0BgNVHR8ELTArMCmgJ6AlhiNodHRw
// SIG // Oi8vY3JsLnZlcmlzaWduLmNvbS9wY2EzLWc1LmNybDA0
// SIG // BggrBgEFBQcBAQQoMCYwJAYIKwYBBQUHMAGGGGh0dHA6
// SIG // Ly9vY3NwLnZlcmlzaWduLmNvbTAdBgNVHSUEFjAUBggr
// SIG // BgEFBQcDAgYIKwYBBQUHAwMwKAYDVR0RBCEwH6QdMBsx
// SIG // GTAXBgNVBAMTEFZlcmlTaWduTVBLSS0yLTgwHQYDVR0O
// SIG // BBYEFM+Zqep7JvRLyY6P1/AFJu/j0qedMB8GA1UdIwQY
// SIG // MBaAFH/TZafC3ey78DAJ80M5+gKvMzEzMA0GCSqGSIb3
// SIG // DQEBBQUAA4IBAQBWIuY0pMRhy0i5Aa1WqGQP2YyRxLvM
// SIG // DOWteqAif99HOEotbNF/cRp87HCpsfBP5A8MU/oVXv50
// SIG // mEkkhYEmHJEUR7BMY4y7oTTUxkXoDYUmcwPQqYxkbdxx
// SIG // kuZFBWAVWVE5/FgUa/7UpO15awgMQXLnNyIGCb4j6T9E
// SIG // mh7pYZ3MsZBc/D3SjaxCPWU21LQ9QCiPmxDPIybMSyDL
// SIG // kB9djEw0yjzY5TfWb6UgvTTrJtmuDefFmvehtCGRM2+G
// SIG // 6Fi7JXx0Dlj+dRtjP84xfJuPG5aexVN2hFucrZH6rO2T
// SIG // ul3IIVPCglNjrxINUIcRGz1UUpaKLJw9khoImgUux5Ol
// SIG // SJHTMYIEcjCCBG4CAQEwgckwgbQxCzAJBgNVBAYTAlVT
// SIG // MRcwFQYDVQQKEw5WZXJpU2lnbiwgSW5jLjEfMB0GA1UE
// SIG // CxMWVmVyaVNpZ24gVHJ1c3QgTmV0d29yazE7MDkGA1UE
// SIG // CxMyVGVybXMgb2YgdXNlIGF0IGh0dHBzOi8vd3d3LnZl
// SIG // cmlzaWduLmNvbS9ycGEgKGMpMTAxLjAsBgNVBAMTJVZl
// SIG // cmlTaWduIENsYXNzIDMgQ29kZSBTaWduaW5nIDIwMTAg
// SIG // Q0ECECnBWz+qzVJqTme9PE5+P/IwCQYFKw4DAhoFAKBw
// SIG // MBAGCisGAQQBgjcCAQwxAjAAMBkGCSqGSIb3DQEJAzEM
// SIG // BgorBgEEAYI3AgEEMBwGCisGAQQBgjcCAQsxDjAMBgor
// SIG // BgEEAYI3AgEVMCMGCSqGSIb3DQEJBDEWBBR8v50qeIPT
// SIG // CjRw4Qg814R/085b9DANBgkqhkiG9w0BAQEFAASCAQCG
// SIG // bgn3/ZoNquPdOMI0g9TsGSjj2MOt3mFdviOX1j6cvdTh
// SIG // 3NYaUqXJBdPuCcuvuYhcgKRgyhdWcoIxlBeqxHkFAAex
// SIG // llqMMcpZ9hnAo9EeJAcr1HcB1SyysOWn8LBnV48yJGUL
// SIG // NB3jEAusiYYgoIJuYOh4Ipi343BIS7EOZr3KbW6PE1cc
// SIG // XZEjrEUj2JDZzM43X8EZANJMx9KhW5RhcmvxDlLocSDS
// SIG // Xmhrm/XXJj6jygwy7NopJBOVBaMfCFfW36tx2O21w68R
// SIG // rNU8PnbaiTYO8LEJ7aZzSczPPkiL7dx3w7CIRhGYr7Ot
// SIG // 0oMpBUElOttIsWoGNWfAX7BsNlAWC/e3oYICCzCCAgcG
// SIG // CSqGSIb3DQEJBjGCAfgwggH0AgEBMHIwXjELMAkGA1UE
// SIG // BhMCVVMxHTAbBgNVBAoTFFN5bWFudGVjIENvcnBvcmF0
// SIG // aW9uMTAwLgYDVQQDEydTeW1hbnRlYyBUaW1lIFN0YW1w
// SIG // aW5nIFNlcnZpY2VzIENBIC0gRzICEA7P9DjI/r81bgTY
// SIG // apgbGlAwCQYFKw4DAhoFAKBdMBgGCSqGSIb3DQEJAzEL
// SIG // BgkqhkiG9w0BBwEwHAYJKoZIhvcNAQkFMQ8XDTE0MDIw
// SIG // NzEwMDkwMFowIwYJKoZIhvcNAQkEMRYEFIGwt6YOXnH6
// SIG // bFquJyrm2H/e9SB6MA0GCSqGSIb3DQEBAQUABIIBAEvo
// SIG // T5m8KzsqRkW5CQxKWHg5keqkVfHfBjGh+O4gxxQuMrmX
// SIG // yAPZNIWbpRQV1pLhcz5F31+jICGrGgRpcZtWHTRpQhxb
// SIG // 9X5Rhc0IuIJI6La3d0nHaQVOwocWLPjP+wfUINUY0qON
// SIG // VpKNCsdIoGSOGVXpfQeiNvU/spjNutw2LP0EN0VG5cOP
// SIG // nbogIcBP29U3icw1TbeSJHiOBoLOHrBLq8d9OvrvjJU/
// SIG // avz5oeOTTwLrE/PmchxKul9MsQo2W4XNKVc+e8yhtOE6
// SIG // t8VveDetw3m/9qMTaC/zoBz7ssPUnpSRqL/VbIinETaT
// SIG // utVURVBNWb+JuvpBDtexkwxBjB5z3q8=
// SIG // End signature block
